﻿//
//  INFO & HELP FILE
//  ----------------
//  This file describes the CPU Register encode and decode funtions 
//  used in the CPU configuration file Intel.xml
//
using System;

namespace NotebookHardwareControl
{
    public class EncodeDecodeInfo
    {
        public class Intel
        {

            #region "Intel Core 2 Duo Desktop Processor class with Half Multiplier support"
            //
            public class IntelCore2DuoDesktop_HM
            {
                #region Multiplier
                ///
                /// Multiplier Register Byte: fmxx-xxxx b
                /// ========================  |||| ||||
                ///                           ||++-++++-- Multiplier Value (0 to 63)
                ///                           |+--------- Half Multiplier (0=no, 1=yes)
                ///                           +---------- SuperLFM (0=no, 1=yes)
                ///
                ///
                /// *********************************************************************************************************
                ///
                ///
                /// Sample 1:  We want to set a multiplier value of 7
                /// ~~~~~~~~   Multiplier = 7 -> CPU_MultiplierId = 0x0E (corresponding stepId of the multiplier value)
                ///
                ///            CPU_MSR_MultiplierID = encodeStepId(CPU_MultiplierId) =
                ///                                 = encodeStepId(0x0E) = 0x07 = f000-0111b
                ///                                                               |||| ||||
                ///                                                               ||++-++++-- Multiplier Value = 00-0111b = 7
                ///                                                               |+--------- Half Multiplier disabled (0=no)
                ///                                                               +---------- not available
                ///
                ///            CPU_MultiplierId = decodeStepId(CPU_MSR_MultiplierID) = decodeStepId(0x07) = 0x0E
                ///                             = 0x0E -> Multiplier = 7 (corresponding value of the stepId)
                ///
                ///            In this Sample 1 Half Multiplier is disabled and so the resulting multiplier in the CPU = 7
                ///
                ///
                ///
                /// Sample 2:  We want to set a multiplier value of 7.5
                /// ~~~~~~~~   Multiplier = 7.5 -> CPU_MultiplierId = 0x0F (corresponding stepId of the multiplier value)
                ///
                ///            CPU_MSR_MultiplierID = encodeStepId(CPU_MultiplierId) =
                ///                                 = encodeStepId(0x0F) = 0x47 = f100-0111b
                ///                                                               |||| ||||
                ///                                                               ||++-++++-- Multiplier Value = 00-0111b = 7
                ///                                                               |+--------- Half Multiplier enabled (1=yes)
                ///                                                               +---------- not available
                ///
                ///            CPU_MultiplierId = decodeStepId(CPU_MSR_MultiplierID) = decodeStepId(0x47) = 0x0F
                ///                             = 0x0F -> Multiplier = 7.5 (corresponding value of the stepId)
                ///
                ///            In this Sample 2 Half Multiplier is enabled and so the resulting multiplier
                ///            in the CPU = 7 + 0.5 = 7.5
                ///
                /// *********************************************************************************************************
                ///
                public class Multiplier
                {
                    /// <summary>
                    /// Convert "NHC MultiplierID" to "CPU MSR MultiplierID"
                    /// </summary>
                    /// <param name="stepId">NHC MultiplierID</param>
                    /// <returns>CPU MSR MultiplierID</returns>
                    public static uint encodeStepId(uint stepId)
                    {
                        if ((stepId % 2) != 0)              // Half Multiplier check
                        {
                            stepId = stepId / 2;      			// halving the stepId to obtain the CPU MSR MultiplierID
                            stepId = stepId | 0x40;     		// Set the Half Multiplier Bit 7
                        }
                        else
                        {
                            stepId = stepId / 2;         		// halving the stepId to obtain the CPU MSR MultiplierID
                        }
                        return stepId;                      // return the CPU MSR MultiplierID for the CPU
                    }
                    /// <summary>
                    /// Convert "CPU MSR MultiplierID" to "NHC MultiplierID"
                    /// </summary>
                    /// <param name="stepId">CPU MSR MultiplierID</param>
                    /// <returns>NHC MultiplierID</returns>
                    public static uint decodeStepId(uint stepId)
                    {
                        if ((stepId & 0x40) != 0)           // check if the Half Multiplier Bit 7 is set
                        {
                            stepId = stepId & 0x3F;  				// remove the Half Multiplier Bit 7
                            stepId = stepId * 2;    				// multiply the msr register value to obtain the multiplier stepId
                            stepId = stepId + 1;        		// add the half multiplier step stepId
                        }
                        else
                        {
                            stepId = stepId * 2;        		// multiply the msr register value to obtain the multiplier stepId
                        }
                        return stepId;                      // return the multiplier stepId
                    }
                }
                //
                #endregion
            }
            //
            #endregion

            #region "Intel Core 2 Duo Mobile Processor class with SuperLFM support"
            //
            public class IntelCore2DuoMobile_SLFM
            {
                #region Multiplier
                ///
                /// Multiplier Register Byte: fmxx-xxxx b
                /// ========================  |||| ||||
                ///                           ||++-++++-- Multiplier Value (0 to 63)
                ///                           |+--------- Half Multiplier (0=no, 1=yes)
                ///                           +---------- SuperLFM (0=no, 1=yes)
                ///
                ///
                /// *********************************************************************************************************
                ///
                ///
                /// Sample 1:  We want to set a multiplier value of 3
                /// ~~~~~~~~   Multiplier = 3 -> CPU_MultiplierId = 0x03 (corresponding stepId of the value)
                ///
                ///            CPU_MSR_MultiplierID = encodeStepId(CPU_MultiplierId) =
                ///                                 = encodeStepId(0x03) = 0x86 = 1m00-0110b
                ///                                                               |||| ||||
                ///                                                               ||++-++++-- Multiplier Value = 00-0110b = 6
                ///                                                               |+--------- Half Multiplier not available
                ///                                                               +---------- SuperLFM enabled (1=yes)
                ///
                ///            CPU_MultiplierId = decodeStepId(CPU_MSR_MultiplierID) = decodeStepId(0x86) = 0x06
                ///                             = 0x03 -> Multiplier = 3 (corresponding value of the stepId)
                ///
                ///            In this Sample 1 SuperLFM (Super Low Frequency Mode) is enabled
                ///            and so the resulting multiplier in the CPU = 6 / 2 = 3
                ///
                ///
                ///
                /// Sample 2:  We want to set a multiplier value of 7
                /// ~~~~~~~~   Multiplier = 7 -> CPU_MultiplierId = 0x07 (corresponding stepId of the multiplier value)
                ///
                ///            CPU_MSR_MultiplierID = encodeStepId(CPU_MultiplierId) =
                ///                                 = encodeStepId(0x07) = 0x07 = 0m00-0111b
                ///                                                               |||| ||||
                ///                                                               ||++-++++-- Multiplier Value = 00-0111b = 7
                ///                                                               |+--------- Half Multiplier not available
                ///                                                               +---------- SuperLFM disabled (0=no)
                ///
                ///            CPU_MultiplierId = decodeStepId(CPU_MSR_MultiplierID) = decodeStepId(0x07) = 0x07
                ///                             = 0x07 -> Multiplier = 7 (corresponding value of the stepId)
                ///
                ///            In this Sample 2 SuperLFM is disabled and so the resulting multiplier in the CPU = 7
                ///
                /// *********************************************************************************************************
                ///
                public class Multiplier
                {
                    /// <summary>
                    /// Convert "NHC MultiplierID" to "CPU MSR MultiplierID"
                    /// </summary>
                    /// <param name="stepId">NHC MultiplierID</param>
                    /// <returns>CPU MSR MultiplierID</returns>
                    public static uint encodeStepId(uint stepId)
                    {
                        if (stepId < 6)             		// enable SuperLFM for "Multiplier < 6" -> "CPU_MultiplierId < 0x06"
                        {
                            stepId = stepId * 2;        // we have to duplicate the multiplier_id if SuperLFM is enabled. If in the CPU SuperLFM is enabled the multiplier_id will be divided by two.
                            stepId = stepId | 0x80;     // Set the SuperLFM Bit 8
                        }
                        return stepId;              		// return the CPU MSR MultiplierID for the CPU
                    }
                    /// <summary>
                    /// Convert "CPU MSR MultiplierID" to "NHC MultiplierID"
                    /// </summary>
                    /// <param name="stepId">CPU MSR MultiplierID</param>
                    /// <returns>NHC MultiplierID</returns>
                    public static uint decodeStepId(uint stepId)
                    {
                        if ((stepId & 0x80) != 0)   		// check if the SuperLFM Bit 8 is set
                        {
                            stepId = stepId & 0x3F;     // remove the SuperLFM Bit 8 and Bit 7 (if set)
                            stepId = stepId / 2;        // If SuperLFM is enabled we have to remove 6 (in encodeStepId() we have added 6 if SuperLFM is enabled)
                        }
                        else
                        {
                            stepId = stepId & 0x3F;     // remove Bit 7 (if set)
                        }
                        return stepId;
                    }
                }
                //
                #endregion
            }
            //
            #endregion

            #region "Intel Core 2 Duo Mobile Processor class with SuperLFM (Super Low Frequency Mode) and Half Multiplier support"
            //
            public class IntelCore2DuoMobile_SLFM_HM
            {
                #region Multiplier
                ///
                /// Multiplier Register Byte: fmxx-xxxx b
                /// ========================  |||| ||||
                ///                           ||++-++++-- Multiplier Value (0 to 63)
                ///                           |+--------- Half Multiplier (0=no, 1=yes)
                ///                           +---------- SuperLFM (0=no, 1=yes)
                ///
                ///
                /// *********************************************************************************************************
                ///
                ///
                /// Sample 1:  We want to set a multiplier value of 3
                /// ~~~~~~~~   Multiplier = 3 -> CPU_MultiplierId = 0x06 (corresponding stepId of the value)
                ///
                ///            CPU_MSR_MultiplierID = encodeStepId(CPU_MultiplierId) =
                ///                                 = encodeStepId(0x06) = 0x86 = 1000-0110b
                ///                                                               |||| ||||
                ///                                                               ||++-++++-- Multiplier Value = 00-0110b = 6
                ///                                                               |+--------- Half Multiplier disabled (0=no)
                ///                                                               +---------- SuperLFM enabled (1=yes)
                ///
                ///            CPU_MultiplierId = decodeStepId(CPU_MSR_MultiplierID) = decodeStepId(0x86) = 0x06
                ///                             = 0x06 -> Multiplier = 3 (corresponding value of the stepId)
                ///
                ///            In this Sample 1 SuperLFM (Super Low Frequency Mode) is enabled
                ///            and so the resulting multiplier in the CPU = 6 / 2 = 3
                ///
                ///
                ///
                /// Sample 2:  We want to set a multiplier value of 6
                /// ~~~~~~~~   Multiplier = 6 -> CPU_MultiplierId = 0x0C (corresponding stepId of the multiplier value)
                ///
                ///            CPU_MSR_MultiplierID = encodeStepId(CPU_MultiplierId) =
                ///                                 = encodeStepId(0x0C) = 0x06 = 000-0110b
                ///                                                               |||| ||||
                ///                                                               ||++-++++-- Multiplier Value = 00-0110b = 6
                ///                                                               |+--------- Half Multiplier disabled (0=no)
                ///                                                               +---------- SuperLFM disabled (0=no)
                ///
                ///            CPU_MultiplierId = decodeStepId(CPU_MSR_MultiplierID) = decodeStepId(0x06) = 0x0C
                ///                             = 0x0C -> Multiplier = 6 (corresponding value of the stepId)
                ///
                ///            In this Sample 2 the SuperLFM and Half Multiplier are disabled
                ///            and so the resulting multiplier in the CPU = 6
                ///
                ///
                ///
                /// Sample 3:  We want to set a multiplier value of 7.5
                /// ~~~~~~~~   Multiplier = 7.5 -> CPU_MultiplierId = 0x0F (corresponding stepId of the multiplier value)
                ///
                ///            CPU_MSR_MultiplierID = encodeStepId(CPU_MultiplierId) =
                ///                                 = encodeStepId(0x0F) = 0x47 = 0100-0111b
                ///                                                               |||| ||||
                ///                                                               ||++-++++-- Multiplier Value = 00-0111b = 7
                ///                                                               |+--------- Half Multiplier enabled (1=yes)
                ///                                                               +---------- SuperLFM disabled (0=no)
                ///
                ///            CPU_MultiplierId = decodeStepId(CPU_MSR_MultiplierID) = decodeStepId(0x47) = 0x0F
                ///                             = 0x0F -> Multiplier = 7.5 (corresponding value of the stepId)
                ///
                ///            In this Sample 3 Half Multiplier is enabled and so the resulting multiplier
                ///            in the CPU = 7 + 0.5 = 7.5
                ///
                /// *********************************************************************************************************
                ///
                public class Multiplier
                {
                    /// <summary>
                    /// Convert "NHC MultiplierID" to "CPU MSR MultiplierID"
                    /// </summary>
                    /// <param name="stepId">NHC MultiplierID</param>
                    /// <returns>CPU MSR MultiplierID</returns>
                    public static uint encodeStepId(uint stepId)
                    {
                        bool superLFM = false;
                        if (stepId < 12)                    // enable SuperLFM for "Multiplier < 6" -> "CPU_MultiplierId < 0x0C (12)"
                        {
                            superLFM = true;
                            stepId = stepId + 6;      			// we have to add 6 if SuperLFM is enabled. If in the CPU SuperLFM is enabled the multiplier_id will be divided by two.
                        }
                        if ((stepId % 2) != 0)              // Half Multiplier check
                        {
                            stepId = stepId / 2;      			// halving the stepId to obtain the CPU MSR MultiplierID
                            stepId = stepId | 0x40;    			// Set the Half Multiplier Bit 7
                        }
                        else
                        {
                            stepId = stepId / 2;      			// halving the stepId to obtain the CPU MSR MultiplierID
                        }
                        if (superLFM == true)
                        {
                            stepId = stepId | 0x80;   			// Set the SuperLFM Bit 8
                        }
                        return stepId;                      // return the CPU MSR MultiplierID for the CPU
                    }
                    /// <summary>
                    /// Convert "CPU MSR MultiplierID" to "NHC MultiplierID"
                    /// </summary>
                    /// <param name="stepId">CPU MSR MultiplierID</param>
                    /// <returns>NHC MultiplierID</returns>
                    public static uint decodeStepId(uint stepId)
                    {
                        bool superLFM = false;
                        if ((stepId & 0x80) != 0)           // check if the SuperLFM Bit 8 is set
                        {
                            stepId = stepId & 0x7F;    			// remove the SuperLFM Bit 8
                            superLFM = true;
                        }
                        if ((stepId & 0x40) != 0)           // check if the Half Multiplier Bit 7 is set
                        {
                            stepId = stepId & 0x3F;     		// remove the Half Multiplier Bit 7
                            stepId = stepId * 2;      			// multiply the msr register value to obtain the multiplier stepId
                            stepId = stepId + 1;       			// add the half multiplier step stepId
                        }
                        else
                        {
                            stepId = stepId * 2;        		// multiply the msr register value to obtain the multiplier stepId
                        }
                        if (superLFM == true)
                        {
                            stepId = stepId - 6;       			// If SuperLFM is enabled we have to remove 6 (in encodeStepId() we have added 6 if SuperLFM is enabled)
                        }
                        return stepId;                      // return the multiplier stepId
                    }
                }
                //
                #endregion
            }
            //
            #endregion

        }
    }
}